﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;

using Microsoft.Practices.Unity;
using VdcCommon;
using VdcCommon.VdcQueries;
using VdcFrontend;
using VdcCommon.Interfaces;
using VdcDAL;
using System.Collections.ObjectModel;
using VdcCommon.VdcAction;
using UI.Common;
using System.ComponentModel;

namespace CWPF
{
	public class ClustersModel : SearchableListModel
	{
		#region Commands

		public Action NewCommand { get; private set; }
		public Action EditCommand { get; private set; }
		public Action RemoveCommand { get; private set; }

		#endregion

		#region Properties

		public ClusterModel NewEditModel { get; set; }
		public RemoveModel RemoveModel { get; set; }


		public bool CanEdit
		{
			get { return (bool)GetValue(CanEditProperty); }
			set { SetValue(CanEditProperty, value); }
		}
		public static readonly DependencyProperty CanEditProperty =
			DependencyProperty.Register("CanEdit", typeof(bool), typeof(ClustersModel), new UIPropertyMetadata(false));


		public bool CanRemove
		{
			get { return (bool)GetValue(CanRemoveProperty); }
			set { SetValue(CanRemoveProperty, value); }
		}
		public static readonly DependencyProperty CanRemoveProperty =
			DependencyProperty.Register("CanRemove", typeof(bool), typeof(ClustersModel), new UIPropertyMetadata(false));

		#endregion


		public ClustersModel(ClustersView view, IUnityContainer container)
			: base(view, container)
		{
			Title = "Clusters";
			SearchStringModel.Value = "Clusters:";

			NewCommand = New;
			EditCommand = Edit;
			RemoveCommand = Remove;

		}

		private void New()
		{
			NewEditModel = Container.Resolve<ClusterModel>();
			NewEditModel.IsOpen = true;
			NewEditModel.Header = "New Cluster";
			NewEditModel.IsNew = true;
			var cpus = DataProvider.GetCPUList();
			NewEditModel.CPUs = cpus;
			NewEditModel.Cpu = cpus.FirstOrDefault();
			NewEditModel.MemoryOverCommit = DataProvider.GetClusterDefaultMemoryOverCommit();
			var storagePools = DataProvider.GetStoragePoolList();
			NewEditModel.StoragePools = storagePools;
			NewEditModel.StoragePool = storagePools.FirstOrDefault();
			NewEditModel.Commands =
				new ArrayList
			        {
			            new
			                {
			                    Command = new DelegateCommand(OnSave),
			                    Text = "OK",
			                    IsDefault = true
			                },
			            new
			                {
			                    Command = new DelegateCommand(Cancel),
			                    Text = "Cancel"
			                }
			        };
		}

		private void Edit()
		{
			VDSGroup cluster = (VDSGroup)SelectedItem;

			NewEditModel = Container.Resolve<ClusterModel>();
			NewEditModel.IsOpen = true;
			NewEditModel.Header = "Edit Cluster";
			NewEditModel.Name = cluster.name;
			NewEditModel.Description = cluster.description;
			var cpus = DataProvider.GetCPUList();
			NewEditModel.CPUs = cpus;
			NewEditModel.Cpu = cpus.FirstOrDefault(a => a.CpuName == cluster.cpu_name);
			NewEditModel.MemoryOverCommit = cluster.max_vds_memory_over_commit;
			var storagePools = DataProvider.GetStoragePoolList();
			NewEditModel.StoragePools = storagePools;
			NewEditModel.StoragePool = storagePools.FirstOrDefault(a => a.id == cluster.storage_pool_id);
			NewEditModel.Commands =
				new ArrayList
			        {
			            new
			                {
			                    Command = new DelegateCommand(OnSave),
			                    Text = "OK",
			                    IsDefault = true
			                },
			            new
			                {
			                    Command = new DelegateCommand(Cancel),
			                    Text = "Cancel"
			                }
			        };
		}

		private void Remove()
		{
			RemoveModel = Container.Resolve<RemoveModel>();
			RemoveModel.IsOpen = true;
			RemoveModel.Header = "Remove Cluster(s)";
			RemoveModel.EntityNames = SelectedItems.Cast<VDSGroup>().Select(a => a.name);
			RemoveModel.Commands =
				new ArrayList
			        {
			            new
			                {
			                    Command = new DelegateCommand(OnRemove),
			                    Text = "OK",
								IsDefault = true
			                },
			            new
			                {
			                    Command = new DelegateCommand(Cancel),
			                    Text = "Cancel"
			                }
			        };
		}

		protected override SearchType SearchType
		{
			get { return SearchType.Cluster; }
		}

		protected override Type[] NestedModelTypes
		{
			get
			{
				return new[] { typeof(ClusterHostsModel), typeof(ClusterDesktopsModel) };
			}
		}

		private void Cancel()
		{
			if (NewEditModel != null && NewEditModel.IsOpen)
			{
				NewEditModel.IsOpen = false;
			}

			if (RemoveModel != null && RemoveModel.IsOpen)
			{
				RemoveModel.IsOpen = false;
			}
		}

		private void OnSave()
		{
			if (!NewEditModel.UpdateSource())
			{
				return;
			}


			VdcReturnValueBase returnValue;

			VDSGroup cluster = NewEditModel.IsNew ? new VDSGroup() : (VDSGroup)SelectedItem;
			cluster.name = NewEditModel.Name;
			cluster.description = NewEditModel.Description;
			cluster.storage_pool_id = NewEditModel.StoragePool.id;
			cluster.cpu_name = NewEditModel.Cpu.CpuName;
			cluster.max_vds_memory_over_commit = NewEditModel.MemoryOverCommit;

			if (NewEditModel.IsNew)
			{
				returnValue = Frontend.RunAction(VdcActionType.AddVdsGroup,
					new VdsGroupOperationParameters(cluster));
			}
			else
			{
				returnValue = Frontend.RunAction(VdcActionType.UpdateVdsGroup,
					new VdsGroupOperationParameters(cluster));
			}

			NewEditModel.IsOpen = returnValue == null || !returnValue.Succeeded;
		}

		private void OnRemove()
		{
			Frontend.RunMultipleActions(VdcActionType.RemoveVdsGroup,
				SelectedItems
				.Cast<VDSGroup>()
				.Select(a => (VdcActionParametersBase)new VdsGroupParametersBase(a.ID))
				.ToList()
			);

			RemoveModel.IsOpen = false;
		}

		public override void OnSelectionChanged(object item, IList items)
		{
			base.OnSelectionChanged(item, items);

			UpdateActionAvailability();
		}

		protected override void ItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.ItemPropertyChanged(sender, e);

			UpdateActionAvailability();
		}

		private void UpdateActionAvailability()
		{
			var items = SelectedItems != null ? SelectedItems.Cast<ISearchable>().ToList() : new List<ISearchable>();

			CanEdit = SelectedItem != null && items.Count == 1;
			CanRemove = items.Count > 0;
		}
	}
}
